//*********************************************************************
// Example3 is a bounded, non-linear optimization problem with 
// five variables and three requirements.
// 
// The goal of the optimization is to minimize the following expression   
//  5.3578547*x[3]*x[3] + 0.8356891*x[1]*x[5] + 37.293239*x[1] - 40792.141
//
//		Subject to the following requirements:
//
//    0.0 <= 85.334407 + 0.0056858*x[2]*x[5] + 0.0006262*x[1]*x[4] - 0.0022053*x[3]*x[5] <=  92.0
//   90.0 <= 80.512490 + 0.0071317*x[2]*x[5] + 0.0029955*x[1]*x[2] + 0.0021813*x[3]*x[3] <= 110.0
//   20.0 <=  9.300961 + 0.0047026*x[3]*x[5] + 0.0012547*x[1]*x[3] + 0.0019085*x[3]*x[4] <=  25.0


//
//	The problem has five continuous decision variables with the following
// ranges:
//
//   78 <= x[1] <= 102
//   33 <= x[2] <= 45
//   27 <= x[3] <= 45
//   27 <= x[4] <= 45
//   27 <= x[5] <= 45
//
// The OptQuest.dll distributed with this example is a demo
// version with the following restrictions:
//		- Maximum variables = 7	
//		- Maximum constraints = 4
//		- Maximum requirements =  4
//		- Maximum iterations = 500
//
//*********************************************************************


#include "ocl.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

void evaluate(double *, double *);

int main(int argc, char **argv)
{
	double ObjVal[4], DecVar[6], Boundary;
	long   i, status, TotalIter, Example, nsol;
	
	if (argc != 3) 
	{
		TotalIter = 500;
		Boundary = 0.5;
	}
	else
	{
		TotalIter = atoi(argv[1]);
		Boundary  = atof(argv[2]);
	}

	if (TotalIter > 500)
	{
		printf("Warning:  The Demo version is limited to 500 iterations\n");
		TotalIter = 500;
	}


	/* Define problem */

	Example = OCLSetup(5, 0, 0, 3,"MIN", 999999999);
	if (Example < 0) 
	{
		printf("OCLSetup error code %d\n", Example);
		exit(1);
	}

	/* Define variables */

	OCLDefineVar(Example, 1, 78, 78.62, 102, "CON", 1);
	OCLDefineVar(Example, 2, 33, 33.44,  45, "CON", 1);
	OCLDefineVar(Example, 3, 27, 31.07,  45, "CON", 1);
	OCLDefineVar(Example, 4, 27, 44.18,  45, "CON", 1);
	OCLDefineVar(Example, 5, 27, 35.22,  45, "CON", 1);

	/* Define requirements */

	OCLDefineReq(Example, 1,  0.0,  92.0);
	OCLDefineReq(Example, 2, 90.0, 110.0);
	OCLDefineReq(Example, 3, 20.0,  25.0);

	/* Generate and evaluate several solutions */
	for (i = 1; i <= TotalIter; i++)
	{
		nsol = OCLGetSolution(Example, DecVar);
		if (nsol < 0) 
		{
			printf("OCLGetSolution error code %d\n", nsol);
			exit(1);
		}
		evaluate(DecVar, ObjVal);
		status = OCLPutSolution(Example, nsol, ObjVal,(double*)OCLGetNull(Example));
		if (status < 0) 
		{
			printf("OCLPutSolution error code %d\n", status);
			exit(1);
		}
		if (!(i%100))
		{
			status = OCLGetBest(Example,DecVar,ObjVal);
			if (status < 0) 
			{
				printf("OCLGetBest error code %d\n", status);
				exit(1);
			}
			printf("Iteration %6d : Best objective function value of %15.7f is %s\n",
				i,ObjVal[0], (status == 0) ? "feasible" : "infeasible");
		}
	}

	/* Display the best solution found */
	status = OCLGetBest(Example,DecVar,ObjVal);
	if (status < 0) 
	{
		printf("OCLGetBest error code %d\n", status);
		exit(1);
	}

	printf("Objective Value = %15.7f\n", ObjVal[0]);

	for (i = 1; i <= 3; i++) printf("Requirement[%2d] = %15.7f\n",i, ObjVal[i]);
	for (i = 1; i <= 5; i++) printf("x[%2d] = %15.7f\n",i, DecVar[i]);

	printf("\nPress any key to end ...\n");
	getchar();
	
	/* Free memory */	
	status = OCLGoodBye(Example);
	if (status < 0) 
	{
		printf("OCLGoodBye error code %d\n", status);
		exit(1);
	}
}

/* Evaluation function */

void evaluate(double *x, double *obj)

{
    obj[0] = 5.3578547*x[3]*x[3] + 0.8356891*x[1]*x[5] + 37.293239*
             x[1] - 40792.141;
	obj[1] = 85.334407 + 0.0056858*x[2]*x[5] + 0.0006262*x[1]*x[4] -
             0.0022053*x[3]*x[5];
	obj[2] = 80.51249 + 0.0071317*x[2]*x[5] + 0.0029955*x[1]*x[2] +
             0.0021813*x[3]*x[3];
    obj[3] = 9.300961 + 0.0047026*x[3]*x[5] + 0.0012547*x[1]*x[3] +
             0.0019085*x[3]*x[4];
}
